<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\User\Follow;

use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Tables\User\UserFollowTable;
use Erlage\Photogram\Data\Dtos\User\UserMetaPushSettingsDTO;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowEnum;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowModel;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowFinder;
use Erlage\Photogram\Data\Models\Notification\NotificationEnum;
use Erlage\Photogram\Data\Models\User\Follow\UserFollowBuilder;
use Erlage\Photogram\Data\Models\Notification\NotificationBuilder;
use Erlage\Photogram\Data\Dtos\Notification\NotificationLinkedContentDTO;

final class UserFollowActions extends ExceptionalRequests
{
    /*
    |--------------------------------------------------------------------------
    | follow user
    |--------------------------------------------------------------------------
    */

    public static function add(): void
    {
        self::userFollowProcessor(true);
    }

    /*
    |--------------------------------------------------------------------------
    | unfollow user
    |--------------------------------------------------------------------------
    */

    public static function remove(): void
    {
        self::userFollowProcessor(false);
    }

    /*
    |--------------------------------------------------------------------------
    | accept pending request
    |--------------------------------------------------------------------------
    */

    public static function accept(): void
    {
        self::userFollowPendingProcessor(true);
    }

    /*
    |--------------------------------------------------------------------------
    | ignore pending request
    |--------------------------------------------------------------------------
    */

    public static function ignore(): void
    {
        self::userFollowPendingProcessor(false);
    }

    /*
    |--------------------------------------------------------------------------
    | follow/unfollow user
    |--------------------------------------------------------------------------
    */

    private static function userFollowProcessor(bool $doFollow): void
    {
        self::process(function () use ($doFollow)
        {
            /*
            |--------------------------------------------------------------------------
            | flags
            |--------------------------------------------------------------------------
            */

            $flagUpdateCache = false;

            $flagSendNotifications = false;

            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $userIdFromReq = self::$request -> findKey(UserTable::ID, RequestTable::PAYLOAD, UserTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure target user exists
            |--------------------------------------------------------------------------
            */

            $targetUserModel = UserModel::findFromId_throwException($userIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | privacy checks
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserAvailable($targetUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | try to get model, this tells whether target user is followed or not
            |--------------------------------------------------------------------------
            */

            $userFollowFinder = (new UserFollowFinder())
                -> setFollowedUserId($targetUserModel -> getId())
                -> setFollowedByUserId(self::$authedUserModel -> getId())
                -> find();

            /*
            |--------------------------------------------------------------------------
            | if un follow the user
            |--------------------------------------------------------------------------
            */

            if (false == $doFollow)
            {
                if ($userFollowFinder -> isFound())
                {
                    $followModel = $userFollowFinder -> popModelFromResults();

                    if ( ! $followModel -> isPending())
                    {
                        $flagUpdateCache = true;
                    }

                    $followModel -> delete();
                }
            }
            /*
            |--------------------------------------------------------------------------
            | else if it's a do follow request
            |--------------------------------------------------------------------------
            */
            elseif (true == $doFollow)
            {
                /*
                |--------------------------------------------------------------------------
                | if already followed
                |--------------------------------------------------------------------------
                */
                if ($userFollowFinder -> isFound())
                {
                    /*
                    |--------------------------------------------------------------------------
                    | add to model to response  map
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(
                        UserFollowTable::getTableName(),
                        $userFollowFinder
                            -> popModelFromResults()
                            -> getDataMap()
                    );
                }

                /*
                |--------------------------------------------------------------------------
                | else if user is not already followed
                |--------------------------------------------------------------------------
                */

                else
                {
                    /*
                    |--------------------------------------------------------------------------
                    | build follow
                    |--------------------------------------------------------------------------
                    */

                    $followModelBuilder = (new UserFollowBuilder())
                        -> setFollowedUserId($targetUserModel -> getId())
                        -> setFollowedByUserId(self::$authedUserModel -> getId());

                    /*
                    |--------------------------------------------------------------------------
                    | if private, mark it pending
                    |--------------------------------------------------------------------------
                    */

                    if ($targetUserModel -> isPrivate())
                    {
                        $followModelBuilder -> setMetaIsPending(UserFollowEnum::META_IS_PENDING_YES);
                    }

                    /*
                    |--------------------------------------------------------------------------
                    | get model
                    |--------------------------------------------------------------------------
                    */

                    $followModel = $followModelBuilder -> dispense();

                    /*
                    |--------------------------------------------------------------------------
                    | do follow
                    |--------------------------------------------------------------------------
                    */

                    $followModel -> save();

                    $flagSendNotifications = true;

                    /*
                    |--------------------------------------------------------------------------
                    | if user count has to be changed
                    |--------------------------------------------------------------------------
                    */

                    if ( ! $followModel -> isPending())
                    {
                        $flagUpdateCache = true;
                    }

                    /*
                    |--------------------------------------------------------------------------
                    |  if everything is alright, add follow to follows map
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(UserFollowTable::getTableName(), $followModel -> getDataMap());
                }
            }

            if ($flagUpdateCache)
            {
                UserModelHelper::updateCacheFollowersCount($targetUserModel);

                UserModelHelper::updateCacheFollowingsCount(self::$authedUserModel);
            }

            /*
            |--------------------------------------------------------------------------
            |  process notifications
            |--------------------------------------------------------------------------
            */

            if ($flagSendNotifications && $doFollow)
            {
                /*
                |--------------------------------------------------------------------------
                |  send a requested notification
                |--------------------------------------------------------------------------
                */

                if ($followModel -> isPending())
                {
                    $linkedContent = new NotificationLinkedContentDTO();

                    $notificationModel = (new NotificationBuilder())
                        -> setToUserId($targetUserModel -> getId())
                        -> setTargetContentId($targetUserModel -> getId())
                        -> setMetaType(NotificationEnum::META_TYPE_USER_FOLLOW_PENDING)
                        -> setMetaIsTransient(NotificationEnum::META_IS_TRANSIENT_YES)
                        -> setLinkedContent($linkedContent)
                        -> dispense();

                    $notificationModel -> save();
                }

                /*
                |--------------------------------------------------------------------------
                |  send a started following notification
                |--------------------------------------------------------------------------
                */
                else
                {
                    $linkedContent = new NotificationLinkedContentDTO();

                    $linkedContent -> linkUserId(self::$authedUserModel -> getId());

                    $notificationModel = (new NotificationBuilder())
                        -> setToUserId($targetUserModel -> getId())
                        -> setTargetContentId($targetUserModel -> getId())
                        -> setMetaType(NotificationEnum::META_TYPE_USER_FOLLOW)
                        -> setLinkedContent($linkedContent)
                        -> dispense();

                    $notificationModel -> save();
                }
            }
        });
    }

    /*
    |--------------------------------------------------------------------------
    | accept/ignore follow request processor
    |--------------------------------------------------------------------------
    */

    private static function userFollowPendingProcessor(bool $doAccept): void
    {
        self::process(function () use ($doAccept)
        {
            /*
            |--------------------------------------------------------------------------
            | flags
            |--------------------------------------------------------------------------
            */

            $flagUpdateCache = false;

            $flagSendNotifications = false;

            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $userFollowIdFromReq = self::$request -> findKey(UserFollowTable::ID, RequestTable::PAYLOAD, UserFollowTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $userFollowIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure target user follow model exists
            |--------------------------------------------------------------------------
            */

            $targetUserFollowModel = UserFollowModel::findFromId_throwException($userFollowIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | ensure ownership
            |--------------------------------------------------------------------------
            */

            if ($targetUserFollowModel -> getFollowedUserId() != self::$authedUserModel -> getId())
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | ensure target user exists
            |--------------------------------------------------------------------------
            */

            $targetUserModel = UserModel::findFromId_throwException($targetUserFollowModel -> getFollowedByUserId());

            /*
            |--------------------------------------------------------------------------
            | if ignore request
            |--------------------------------------------------------------------------
            */

            if (false == $doAccept)
            {
                $targetUserFollowModel -> delete();

                $flagUpdateCache = true;
            }

            /*
            |--------------------------------------------------------------------------
            | else if it's a accept follow request
            |--------------------------------------------------------------------------
            */

            elseif (true == $doAccept)
            {
                /*
                |--------------------------------------------------------------------------
                | if already followed
                |--------------------------------------------------------------------------
                */
                if ( ! $targetUserFollowModel -> isPending())
                {
                    /*
                    |--------------------------------------------------------------------------
                    | add to model to response  map
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(UserFollowTable::getTableName(), $targetUserFollowModel -> getDataMap());
                }

                /*
                |--------------------------------------------------------------------------
                | else if user is not already accepted
                |--------------------------------------------------------------------------
                */

                else
                {
                    /*
                    |--------------------------------------------------------------------------
                    | update follow pending status
                    |--------------------------------------------------------------------------
                    */

                    $targetUserFollowModel -> update(
                        array(
                            UserFollowTable::META_IS_PENDING => UserFollowEnum::META_IS_PENDING_NO,
                        )
                    );

                    $targetUserFollowModel -> save();

                    $flagUpdateCache = true;

                    $flagSendNotifications = true;

                    /*
                    |--------------------------------------------------------------------------
                    |  if everything is alright, add model to response
                    |--------------------------------------------------------------------------
                    */

                    self::addToResponse(UserFollowTable::getTableName(), $targetUserFollowModel -> getDataMap());
                }
            }

            if ($flagUpdateCache)
            {
                // reverse order

                UserModelHelper::updateCacheFollowersCount(self::$authedUserModel);

                UserModelHelper::updateCacheFollowingsCount($targetUserModel);
            }

            /*
            |--------------------------------------------------------------------------
            |  process notifications
            |--------------------------------------------------------------------------
            */

            if ($flagSendNotifications && $doAccept)
            {
                /*
                |--------------------------------------------------------------------------
                |  send a started following notification to current user
                |--------------------------------------------------------------------------
                */

                $linkedContent = new NotificationLinkedContentDTO();

                $linkedContent -> linkUserId($targetUserModel -> getId());

                $notificationModel = (new NotificationBuilder())
                    -> setToUserId(self::$authedUserModel -> getId())
                    -> setTargetContentId(self::$authedUserModel -> getId())
                    -> setMetaType(NotificationEnum::META_TYPE_USER_FOLLOW)
                    -> setLinkedContent($linkedContent)
                    -> dispense();

                $notificationModel -> save();

                /*
                |--------------------------------------------------------------------------
                |  send a accepted follow request notification to target user
                |--------------------------------------------------------------------------
                */

                $targetUserMetaPushSettings = $targetUserModel -> getMetaPushSettings();

                if (UserMetaPushSettingsDTO::VALUE_ON == $targetUserMetaPushSettings -> getAcceptedFollowRequest())
                {
                    $linkedContent = new NotificationLinkedContentDTO();

                    $linkedContent -> linkUserId(self::$authedUserModel -> getId());

                    $notificationModel = (new NotificationBuilder())
                        -> setToUserId($targetUserModel -> getId())
                        -> setTargetContentId(self::$authedUserModel -> getId())
                        -> setMetaType(NotificationEnum::META_TYPE_USER_FOLLOW_ACCEPTED)
                        -> setLinkedContent($linkedContent)
                        -> dispense();

                    $notificationModel -> save();
                }
            }
        });
    }
}
